/**
 * \file sdc_advanced.h
 *
 * \brief Definition of some advanced libSDC functions - normal applications
 * must not use these functions.
 *
 * <b>SDC private header</b>
 *
 * \author Christoph Gellner (cgellner@de.adit-jv.com)
 *
 * \copyright (c) 2015 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 *
 ***********************************************************************/

/**
 * \defgroup add_libSDC_API Additional API
 * Additionally exported function.
 * These functions are not required by normal applications using libSDC.
 */

/**
 * \defgroup advanced Advanced
 * \ingroup add_libSDC_API
 * Advanced libSDC functions. Normal application must not use these functions.
 */

#ifndef __SDC_LIB_ADVANCED_H_
#define __SDC_LIB_ADVANCED_H_

#ifdef __cplusplus
extern "C"
{
#endif

#include <sdc.h>

/* Definitions types and defaults */

/**
 * \ingroup advanced
 * \brief Used to specify that this function is part of the advanced API of libSDC
 *
 * Usual applications using libSDC won't use these functions.
 * Anyway these functions need to be visible for e.g. the SDC daemon or similar */
#define SDC_ADV_API __attribute__((__visibility__("default")))

/**
 * \ingroup advanced
 * \brief Config entry to reference keystore path
 */
#define SDC_CONFIG_KEYSTORE "key_store"
/**
 * \ingroup advanced
 * \brief Config entry to reference keycontainer path
 */
#define SDC_CONFIG_CONTAINER "key_container"

/* Functions */

/**
 * \ingroup advanced
 * \brief Lookup config value in sdc config file
 *
 * Search for the key named \p lookup in the configuration file and return the
 * value using \p val
 *
 * \param[out] val       val to return
 * \param[in]  lookup    key to look up
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - val or lookup are invalid (i.e. NULL)
 * \return \ref SDC_NO_MEM - failed to allocate memory to buffer libSDC config
 * \return \ref SDC_CONFIG_ACCESS_FAILED - failed to read libSDC config
 * \return \ref SDC_CONFIG_MISSING - lookup item not found in libSDC config
 */
SDC_ADV_API sdc_error_t sdc_config_file_lookup(char **val, const char *lookup);


/**
 * \ingroup advanced
 * \brief check if it is an absolute path
 *
 * \param[in] path       path to check
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_CONFIG_INVALID - absolute path checks failed
 */
SDC_ADV_API sdc_error_t sdc_config_is_absolute_path(const char *path);


/**
 * \ingroup advanced
 * \brief Lookup config value in sdc config file and check if it is an absolute path
 *
 * Search for the key named \p lookup in the configuration file,
 * check if it is an absolute path and return the
 * value using \p val
 *
 * \param[out] val       val to return
 * \param[in]  lookup    key to look up
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_INVALID_PARAMETER - val or lookup are invalid (i.e. NULL)
 * \return \ref SDC_NO_MEM - failed to allocate memory to buffer libSDC config
 * \return \ref SDC_CONFIG_ACCESS_FAILED - failed to read libSDC config
 * \return \ref SDC_CONFIG_MISSING - lookup item not found in libSDC config
 * \return \ref SDC_CONFIG_INVALID - absolute path checks failed
 */
SDC_ADV_API sdc_error_t sdc_config_file_lookup_absolute_path(char **val, const char *lookup);


/**
 * \ingroup advanced
 * \brief Verify the kernel version of the libSDC kernel interface
 *
 * If \p version_info_str is not NULL this function will internally
 * allocate some memory to store the version info.
 * This allocated memory needs to be freed by the calling process
 *
 * \param[out] version_info_str  optional pointer to return the version info
 *
 * \return \ref SDC_OK - no error
 * \return \ref SDC_NO_MEM - failed to allocate memory required for error message
 * \return \ref SDC_INVALID_VERSION version of the kernel interface used by libSDC does not fit to the current user-space library version
 * \return otherwise an unexpected error occurred that should be handled by default
 */
SDC_ADV_API sdc_error_t sdc_kernel_version_verify(char **version_info_str);


#ifdef __cplusplus
}
#endif

#endif
